#include "Ymdeltat.h"
#include <stdlib.h>
#include <string.h>


#define ADPCM_VOLUME     356

// Bitmask for register 0x07
#define R07_RESET        0x01
#define R07_SP_OFF       0x08
#define R07_REPEAT       0x10
#define R07_MEMORY_DATA  0x20
#define R07_REC          0x40
#define R07_START        0x80

//Bitmask for register 0x08
#define R08_ROM          0x01
#define R08_64K          0x02
#define R08_DA_AD        0x04
#define R08_SAMPL        0x08
#define R08_NOTE_SET     0x40
#define R08_CSM          0x80

#define DMAX  0x6000
#define DMIN  0x7F
#define DDEF  0x7F

#define DECODE_MAX  32767
#define DECODE_MIN  -32768

#define GETA_BITS    14
#define MAX_STEP     (1<<(16+GETA_BITS))

#define CLAP(min, x, max) ((x < min) ? min : ((max < x) ? max : x))


static void adpcmRestart(Adpcm* adpcm)
{
	adpcm->playAddr     = adpcm->startAddr & adpcm->addrMask;
	adpcm->nowStep      = MAX_STEP - adpcm->step;
	adpcm->out          = 0;
    adpcm->output       = 0;
	adpcm->diff         = DDEF;
	adpcm->nextLeveling = 0;
	adpcm->sampleStep   = 0;
	adpcm->volumeWStep  = (int)((double)adpcm->volume * adpcm->step / MAX_STEP);
}

Adpcm* adpcmCreate(int ramSize, void* y8950, UInt32 systemTime)
{
    Adpcm* adpcm = calloc(1, sizeof(Adpcm));
    adpcm->ramBank = malloc(ramSize);
    adpcm->y8950 = y8950;
    memset(adpcm->ramBank, 0xff, ramSize);
    adpcm->ramSize = ramSize;

    return adpcm;
}

void adpcmDestroy(Adpcm* adpcm)
{
    free(adpcm->ramBank);
    free(adpcm);
}

void adpcmReset(Adpcm* adpcm, UInt32 systemTime)
{
	adpcm->playing   = 0;
	adpcm->startAddr = 0;
	adpcm->stopAddr  = 7;
	adpcm->memPntr   = 0;
	adpcm->delta     = 0;
	adpcm->step      = 0;
	adpcm->addrMask  = (1 << 19) - 1;
	adpcm->reg7      = 0;
	adpcm->reg15     = 0;

	adpcmWriteReg(adpcm, 0x12, 0xff, systemTime);
	
    adpcmRestart(adpcm);
}

void adpcmWriteReg(Adpcm* adpcm, UInt8 rg, UInt8 data, UInt32 systemTime)
{
	switch (rg) { 
		case 0x07: // START/REC/MEM DATA/REPEAT/SP-OFF/-/-/RESET
			adpcm->reg7 = data;
			if (adpcm->reg7 & R07_RESET) {
				adpcm->playing = 0;
			} 
            else if (data & R07_START) {
				adpcm->playing = 1;
				adpcmRestart(adpcm);
			}
			break;

		case 0x08: // CSM/KEY BOARD SPLIT/-/-/SAMPLE/DA AD/64K/ROM 
			adpcm->romBank = data & R08_ROM;
			adpcm->addrMask = data & R08_64K ? (1<<17)-1 : (1<<19)-1;
			break;

		case 0x09: // START ADDRESS (L)
			adpcm->startAddr = (adpcm->startAddr & 0x7F800) | (data << 3);
			adpcm->memPntr = 0;
			break;
		case 0x0A: // START ADDRESS (H) 
			adpcm->startAddr = (adpcm->startAddr & 0x007F8) | (data << 11);
			adpcm->memPntr = 0;
			break;

		case 0x0B: // STOP ADDRESS (L)
			adpcm->stopAddr = (adpcm->stopAddr & 0x7F807) | (data << 3);
			break;
		case 0x0C: // STOP ADDRESS (H) 
			adpcm->stopAddr = (adpcm->stopAddr & 0x007FF) | (data << 11);
			break;

		case 0x0F: // ADPCM-DATA
			//if ((reg7 & R07_REC) && (reg7 & R07_MEMORY_DATA)) {
			{
				int tmp = ((adpcm->startAddr + adpcm->memPntr) & adpcm->addrMask) / 2;
				tmp = (tmp < adpcm->ramSize) ? tmp : (tmp & (adpcm->ramSize - 1)); 
				if (!adpcm->romBank) {
					adpcm->ramBank[tmp] = data;
				}
				adpcm->memPntr += 2;
				if ((adpcm->startAddr + adpcm->memPntr) > adpcm->stopAddr) {
                    adpcmSetStatus(adpcm->y8950, 0x10);
				}
//                adpcmSetStatus(adpcm->y8950, 0x08);
			}
			break;

		case 0x10: // DELTA-N (L) 
			adpcm->delta = (adpcm->delta & 0xFF00) | data;
			adpcm->step = (UInt32)((double)(adpcm->delta << GETA_BITS) * 3579545 / 72 / 44100 + 0.5);
			adpcm->volumeWStep = (int)((double)adpcm->volume * adpcm->step / MAX_STEP);
			break;
		case 0x11: // DELTA-N (H) 
			adpcm->delta = (adpcm->delta & 0x00FF) | (data << 8);
			adpcm->step = (UInt32)((double)(adpcm->delta << GETA_BITS) * 3579545 / 72 / 44100 + 0.5);
			adpcm->volumeWStep = (int)((double)adpcm->volume * adpcm->step / MAX_STEP);
			break;

		case 0x12: { // ENVELOP CONTROL 
			int oldVol = adpcm->volume;
			adpcm->volume = (data * ADPCM_VOLUME) >> 8;
			if (oldVol != 0) {
				double factor = (double)adpcm->volume / (double)oldVol;
				adpcm->output =     (int)((double)adpcm->output     * factor);
				adpcm->sampleStep = (int)((double)adpcm->sampleStep * factor);
			}
			adpcm->volumeWStep = (int)((double)adpcm->volume * adpcm->step / MAX_STEP);
			break;
		}
		case 0x0D: // PRESCALE (L) 
		case 0x0E: // PRESCALE (H) 
		case 0x15: // DAC-DATA  (bit9-2)
		case 0x16: //           (bit1-0)
		case 0x17: //           (exponent)
		case 0x1A: // PCM-DATA
			// not implemented
			break;
	}
}


UInt8 adpcmReadReg(Adpcm* adpcm, UInt8 rg)
{
	UInt8 result;
	switch (rg) {
		case 0x0F: { // ADPCM-DATA
			// TODO don't advance pointer when playing???
			int adr = ((adpcm->startAddr + adpcm->memPntr) & adpcm->addrMask) / 2;
			if (adpcm->romBank || (adr >= adpcm->ramSize)) {
				result = 0xFF;
			} else {
				result = adpcm->ramBank[adr];
			}
			adpcm->memPntr += 2;
			if ((adpcm->startAddr + adpcm->memPntr) > adpcm->stopAddr) {
                adpcmSetStatus(adpcm->y8950, 0x10);
			}
			break;
		}
		case 0x13: // TODO check
			result = adpcm->out & 0xFF;
			break;
		case 0x14: // TODO check
			result = adpcm->out / 256;
			break;
		default:
			result = 255;
	}
	return result;
}


int adpcmCalcSample(Adpcm* adpcm)
{
	// This table values are from ymdelta.c by Tatsuyuki Satoh.
	static const int F1[16] = { 1,   3,   5,   7,   9,  11,  13,  15,
				   -1,  -3,  -5,  -7,  -9, -11, -13, -15};
	static const int F2[16] = {57,  57,  57,  57,  77, 102, 128, 153,
				   57,  57,  57,  57,  77, 102, 128, 153};
	
	if ((!adpcm->playing) || (adpcm->reg7 & R07_SP_OFF)) {
		return 0;
	}
	adpcm->nowStep += adpcm->step;
	if (adpcm->nowStep >= MAX_STEP) {
		int nowLeveling;
		do {
            int deltaNext;
            int prevOut;
            UInt32 val;
            
			adpcm->nowStep -= MAX_STEP;

            if (!(adpcm->playAddr & 1)) {
				// n-th nibble
				int tmp = adpcm->playAddr / 2;
				if (adpcm->romBank || (tmp >= adpcm->ramSize)) {
					adpcm->reg15 = 0xFF;
				} else {
					adpcm->reg15 = adpcm->ramBank[tmp];
				}
				val = adpcm->reg15 >> 4;
			} else {
				val = adpcm->reg15 & 0x0F;
			}
			prevOut = adpcm->out;
			adpcm->out = CLAP(DECODE_MIN, adpcm->out + (adpcm->diff * F1[val]) / 8, DECODE_MAX);
			adpcm->diff = CLAP(DMIN, (adpcm->diff * F2[val]) / 64, DMAX);
			deltaNext = adpcm->out - prevOut;
			nowLeveling = adpcm->nextLeveling;
			adpcm->nextLeveling = prevOut + deltaNext / 2;
		
			adpcm->playAddr++;
			if (adpcm->playAddr > adpcm->stopAddr) {
				if (adpcm->reg7 & R07_REPEAT) {
					adpcmRestart(adpcm);
				} else {
                    adpcmSetStatus(adpcm->y8950, 0x10);
					adpcm->playing = 0;
				}
			}
		} while (adpcm->nowStep >= MAX_STEP);
		adpcm->sampleStep = (adpcm->nextLeveling - nowLeveling) * adpcm->volumeWStep;
		adpcm->output = nowLeveling * adpcm->volume;
		adpcm->output += (int)((double)adpcm->sampleStep * ((double)adpcm->nowStep/(double)adpcm->step));
	}
	adpcm->output += adpcm->sampleStep;
	return adpcm->output;
}
